<?php

namespace App\Http\Controllers;

use App\Models\Photo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PhotoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $photos = Photo::where('deleted', false)->get();
        return view('photo.index', compact('photos'));
        // return response()->json($photo);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        return view('photo.create');
    }

    public function store(Request $request)
    {
        // Fazer o upload da imagem para storage/public/photos
        try {


            $request['deleted'] =  0;
            $request['users_id'] =  auth()->id();

            $validatedData = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'users_id' => 'required|integer|exists:users,id',
                'deleted' => 'integer',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,mp4|max:2048' // Imagem opcional
            ]);

            // Capturar o ficheiro enviado
            $image = $request->file('image');

            // Obter o nome original do ficheiro
            $originalName = $image->getClientOriginalName();

            // Fazer o upload da imagem para storage/public/photos
            $path = $image->store('photos', 'public');

            $photo =   Photo::create([
                'users_id' => auth()->id(), // Supondo que estás a usar autenticação
                'title' => $request->title,
                'description' => $request->description,
                'image' => $path,
                'deleted' => 0,
                'original_name' => $originalName, // Armazenar o nome original do ficheiro
            ]);



            return response()->json([
                'message' => 'Foto carregada com sucesso.',
                'data' => $photo,
                'response' => 201
            ], 201);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Erro de validação', 'messages' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }



    /**
     * Display the specified resource.
     */
    public function show(Photo $photo)
    {
        try {


            $photo = Photo::findOrFail($photo->id);

            return view('photo.show', compact('photo'));

            if (!$photo) {
                return response()->json(['error' => 'Foto não encontrada.'], 404);
            }

            return response()->json($photo);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Foto não encontrada'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Photo $photo)
    {
        $photo = Photo::findOrFail($photo->id);
        return view('photo.edit', compact('photo'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Photo $photo)
    {
        try {
            // Adiciona campos necessários ao request
            $request['deleted'] =  0;
            $request['users_id'] = auth()->id();

            // Validação dos dados
            $validatedData = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'users_id' => 'required|integer|exists:users,id',
                'deleted' => 'integer',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,mp4|max:2048' // Imagem opcional
            ]);

            // Inicialmente, mantém o caminho da imagem existente
            $path = $photo->image;
            $originalName = $photo->original_name;

            // Se houver nova imagem, faz o upload e apaga a imagem antiga
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $originalName = $image->getClientOriginalName(); // Nome original do ficheiro

                // Fazer upload da nova imagem para storage/public/photos
                $path = $image->store('photos', 'public');

                // Apagar a imagem antiga, se houver
                if ($photo->image) {
                    Storage::disk('public')->delete($photo->image);
                }
            }

            // Atualizar os dados da foto
            $photo->update([
                'users_id' => auth()->id(),
                'title' => $request->title,
                'description' => $request->description,
                'image' => $path, // Usa a nova imagem ou mantém a antiga
                'deleted' => 0,
            ]);

            return response()->json([
                'message' => 'Foto atualizada com sucesso.',
                'data' => $photo,
                'response' => 201
            ], 200);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Erro de validação', 'messages' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Photo $photo)
    {
        try {
            $photo = Photo::findOrFail($photo->id);

            $photo->update(['deleted' => 1]);

            return redirect()->route('photo.index')->with('success', 'Foto excluída com sucesso');

            /* return response()->json([
                    'message' => 'Foto excluída com sucesso.',
                    'data' => $photo
                ]);*/
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    public function getPhotos(Photo $photo)
    {
        try {
            $photos = $photo->where('deleted', false)
                ->get();

            return response()->json($photos);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Nenhuma foto foi encontrada.'], 404);
        }
    }


    public function getPhoto($id)
    {
        try {
            $photos = Photo::where('deleted', false)->where('id', $id)
                ->get();

            return response()->json($photos);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Foto não encontrada.'], 404);
        }
    }
}
