<?php

namespace App\Http\Controllers;

use App\Models\News;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Storage;

class NewsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $news = News::where('deleted', false)->get();

        return view('news.index', compact('news'));

        // return response()->json($news);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        return view('news.create');
    }



    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        $request['deleted'] = 0;

        try {
            $validatedData = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'users_id' => 'required|integer|exists:users,id',
                'deleted' => 'integer',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,mp4|max:2048' // Imagem opcional
            ]);


            // Capturar o ficheiro enviado
            $image = $request->file('image');

            // Obter o nome original do ficheiro
            $originalName = $image->getClientOriginalName();

            // Fazer o upload da imagem para storage/public/photos
            $path = $image->store('news', 'public');

            $news = News::create([
                'users_id' => auth()->id(), // Supondo que estás a usar autenticação
                'title' => $request->title,
                'description' => $request->description,
                'image' => $path,
                'deleted' => 0,
                'original_name' => $originalName, // Armazenar o nome original do ficheiro
            ]);

            return response()->json([
                'message' => 'Post criado com sucesso.',
                'data' => $news,
                'response' => 201
            ], 201);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Erro de validação', 'messages' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(News $news)
    {
        try {

            $news = News::findOrFail($news->id);
            return view('news.show', compact('news'));

            if (!$news) {
                return response()->json(['error' => 'Post não encontrado.'], 404);
            }

            return response()->json($news);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Post não encontrado'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(News $news)
    {
        $news = News::findOrFail($news->id);
        return view('news.edit', compact('news'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, News $news)
    {
        try {
            $news = News::findOrFail($news->id);

            if ($news->deleted) {
                return response()->json(['error' => 'Post não encontrado.'], 404);
            }

            $validatedData = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'users_id' => 'required|integer|exists:users,id',
                'deleted' => 'integer',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,mp4|max:2048' // Imagem opcional
            ]);

            // Inicialmente, mantém o caminho da imagem existente
            $path = $news->image;
            $originalName = $news->original_name;

            // Se houver nova imagem, faz o upload e apaga a imagem antiga
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $originalName = $image->getClientOriginalName(); // Nome original do ficheiro

                // Fazer upload da nova imagem para storage/public/photos
                $path = $image->store('news', 'public');

                // Apagar a imagem antiga, se houver
                if ($news->image) {
                    Storage::disk('public')->delete($news->image);
                }
            }


            $news->update(
                [
                    'users_id' => auth()->id(), // Supondo que estás a usar autenticação
                    'title' => $request->title,
                    'description' => $request->description,
                    'image' => $path,
                    'deleted' => 0,
                    'original_name' => $originalName, // Armazenar o nome original do ficheiro
                ]
            );

            return response()->json([
                'message' => 'Post atualizado com sucesso.',
                'data' => $news,
                'response' => 200
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Erro de validação', 'messages' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(News $news)
    {
        try {
            $news = News::findOrFail($news->id);

            $news->update(['deleted' => 1]);

            return redirect()->route('news.index');

            return response()->json(['message' => 'Post excluído com sucesso.']);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    public function getNews(News $news)
    {
        try {
            $news = $news->where('deleted', false)
                ->get();

            return response()->json($news);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Nenhuma foto foi encontrada.'], 404);
        }
    }


    public function getNew($id)
    {
        try {
            $new = News::where('deleted', false)->where('id', $id)
                ->get();

            return response()->json($new);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Foto não encontrada.'], 404);
        }
    }
}
